#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
Pytheia100_main.py — v1.0.0
"""

import os, sys, json, importlib
from pathlib import Path
from datetime import datetime as dt, timezone

def _enable_vt_win():
    try:
        if os.name != "nt":
            return
        import ctypes
        kernel32 = ctypes.windll.kernel32
        for std in (-11, -12):  # stdout, stderr
            h = kernel32.GetStdHandle(std)
            if h in (0, -1):
                continue
            mode = ctypes.c_uint32()
            if kernel32.GetConsoleMode(h, ctypes.byref(mode)):
                kernel32.SetConsoleMode(h, mode.value | 0x0004)
    except Exception:
        pass

try:
    _enable_vt_win()
    import colorama  
    colorama.init(convert=True, strip=False)
except Exception:
    pass

PRODUCT = "Pytheia100"
VERSION = "1.0.0"

HERE = Path(__file__).resolve().parent
MAIN_FILE = HERE / "Pytheia100_main.py"
INTEGRITY_JSON = HERE / "INTEGRITY.json"
INTEGRITY_SIG  = HERE / "INTEGRITY.sig"
PUBLIC_KEY_PEM = HERE / "pytheia_public_key.pem"
LICENSE_JSON   = HERE / "pytheia_license.json"

from pytheia_guard import (
    verify_code_integrity,
    verify_manifest_signature,
    license_boot,
)

_manifest_obj = verify_manifest_signature(str(INTEGRITY_JSON), str(INTEGRITY_SIG), str(PUBLIC_KEY_PEM))
expected_sha = None
if isinstance(_manifest_obj, dict):
    if "files" in _manifest_obj:
        expected_sha = _manifest_obj["files"].get("Pytheia100_main.py")
    elif "Pytheia100_main.py" in _manifest_obj:
        expected_sha = _manifest_obj["Pytheia100_main.py"]
try:
    if expected_sha:
        verify_code_integrity(str(MAIN_FILE), expected_sha)
except Exception as e:
    print(f"[WARN] Main integrity check: {e}")

LICENSE = license_boot(
    license_path=str(LICENSE_JSON),
    public_key_path=str(PUBLIC_KEY_PEM),
    expected_product=PRODUCT,
    check_machine_hash=False,
)
ORG_NAME = LICENSE.get("org", "Unknown Organization")

sys.path.insert(0, str(HERE))
cfg = importlib.import_module("config")
core = importlib.import_module("Pytheia100_core")

try:
    if os.environ.get("PYTHEIA_COLOR", "auto").lower() == "never":
        def _mono(s, *a, **k): return s
        try: setattr(core, "colored", _mono)
        except: pass
        try:
            import termcolor as _tc
            _tc.colored = _mono
        except: pass
except Exception:
    pass

core.input_dat      = getattr(cfg, "input_dat", core.input_dat)
core.selected_terms = list(getattr(cfg, "selected_terms", core.selected_terms))
core.output_dir     = getattr(cfg, "output_dir", core.output_dir)
core.mode           = getattr(cfg, "mode", core.mode)
core.lat            = getattr(cfg, "lat", core.lat)
core.aux1           = getattr(cfg, "aux1", core.aux1)
core.aux2           = getattr(cfg, "aux2", core.aux2)
Path(core.output_dir).mkdir(parents=True, exist_ok=True)

audit_path = Path(core.output_dir) / "pytheia_audit.jsonl"
def _audit(event: str, **kwargs):
    rec = {"ts": dt.now(timezone.utc).isoformat(), "event": event}
    rec.update(kwargs)
    try:
        with open(audit_path, "a", encoding="utf-8") as f:
            f.write(json.dumps(rec) + "\n")
    except Exception as e:
        print(f"[WARN] audit write failed: {e}")

print(f"[INFO] {PRODUCT} v{VERSION} | Org: {ORG_NAME}")
print(f"[INFO] Mode: {core.mode} | Data: {core.input_dat}")
print(f"[INFO] Selected terms ({len(core.selected_terms)}): {core.selected_terms}")
print(f"[INFO] Output directory: {core.output_dir}")
abs_outdir = str(Path(core.output_dir).resolve())
print(f"PYTHEIA:OUTPUT_DIR={abs_outdir}")
try:
    (HERE / "pytheia_outdir.txt").write_text(abs_outdir, encoding="utf-8")
except Exception as e:
    print(f"[WARN] could not write pytheia_outdir.txt: {e}")
_audit("startup", org=ORG_NAME, mode=core.mode,
       data=os.path.basename(core.input_dat),
       outdir=core.output_dir, terms=core.selected_terms)

try:
    setattr(core, "org_name", ORG_NAME)
    setattr(core, "version_string", f"{PRODUCT} v{VERSION}")
    setattr(core, "report_timestamp", dt.now(timezone.utc).strftime("%Y-%m-%d %H:%M:%SZ"))

    core.run_pointing_model(core.input_dat, core.output_dir, core.selected_terms)
    _audit("run-complete", status="ok")
except SystemExit:
    _audit("run-complete", status="sys-exit")
    raise
except Exception as e:
    _audit("run-failed", status="error", error=str(e))
    raise

print("[INFO] Done.")
