#!/usr/bin/env python3
# Pytheia100_main.py — v1.1.0

from __future__ import annotations
import os
import sys
import json
import importlib
import inspect
import math
import hashlib
from pathlib import Path
from datetime import datetime, timezone
import traceback
import numpy as np

from pytheia_guard import (
    license_boot,
    verify_code_integrity,
    audit_event,
    seat_lock,
)

try:
    import colorama
    colorama.init(convert=True, strip=False)
except Exception:
    pass

try:
    if hasattr(sys.stdout, "reconfigure"):
        sys.stdout.reconfigure(encoding="utf-8", errors="replace")
    if hasattr(sys.stderr, "reconfigure"):
        sys.stderr.reconfigure(encoding="utf-8", errors="replace")
except Exception:
    pass

PRODUCT = "Pytheia100"
VERSION = "1.1.0"

def _parse_expiry(lic: dict):
    ts = lic.get("expires_at") or lic.get("exp") or lic.get("expires")
    if not ts:
        return None, None
    try:
        s = str(ts).strip()
        if s.endswith("Z"):
            s = s[:-1] + "+00:00"
        dt = datetime.fromisoformat(s)
        if dt.tzinfo is None:
            dt = dt.replace(tzinfo=timezone.utc)
        now = datetime.now(timezone.utc)
        delta = (dt - now).total_seconds() / 86400.0
        return dt.astimezone(timezone.utc), delta
    except Exception:
        return None, None

def _derive_phi_from_lat(lat):
    try:
        if lat is None:
            return None
        return math.radians(float(lat))
    except Exception:
        return None

def _unique_output_dir(base_dir: str, selected_terms) -> str:
    base_dir = base_dir or "."
    Path(base_dir).mkdir(parents=True, exist_ok=True)
    key = ",".join(selected_terms or []).encode("utf-8")
    stem = f"terms_{hashlib.sha1(key).hexdigest()[:8]}"
    candidate = Path(base_dir) / stem
    if not candidate.exists():
        candidate.mkdir(parents=True, exist_ok=True)
        return str(candidate)
    i = 2
    while True:
        c = Path(base_dir) / f"{stem}-{i}"
        if not c.exists():
            c.mkdir(parents=True, exist_ok=True)
            return str(c)
        i += 1

import re
_ANSI_RE = re.compile(r"\x1B\[[0-?]*[ -/]*[@-~]")

def _ansi_stripper(text: str) -> str:
    return _ANSI_RE.sub("", str(text))

def supports_ansi() -> bool:
    if not sys.stdout.isatty():
        return False
    if os.name != "nt":
        return True
    if os.environ.get("WT_SESSION") or os.environ.get("ANSICON") or os.environ.get("ConEmuANSI") == "ON":
        return True
    term = os.environ.get("TERM")
    if term and term.lower() not in ("dumb", "unknown"):
        return True
    return False

LICENSE = license_boot(
    license_path="pytheia_license.json",
    public_key_path="pytheia_public_key.pem",
    expected_product=PRODUCT,
    expected_org=None,
    check_machine_hash=False,
)

_org = LICENSE.get("org", LICENSE.get("org_name", "-")) or "-"
_exp_dt, _days_left = _parse_expiry(LICENSE)
if _exp_dt and _days_left is not None and _days_left <= 7.0:
    print(f"[WARNING] License for {_org} expires in {max(_days_left,0):.1f} days ({_exp_dt.strftime('%Y-%m-%dT%H:%M:%SZ')}).")
    print("          Contact support@orionisgroup.com for renewal.")

_here = Path(__file__).resolve().parent
with open(_here / "INTEGRITY.json", "r", encoding="utf-8-sig") as _f:
    _integrity = json.load(_f)
EXPECTED_MAIN_SHA256 = _integrity.get("Pytheia100_main.py")
verify_code_integrity(str(_here / "Pytheia100_main.py"), EXPECTED_MAIN_SHA256)

import config as cfg
import Pytheia100_core as core

def _ensure_colored(core_module):
    ansi_ok = supports_ansi()
    try:
        from termcolor import colored as _tc_colored
        if ansi_ok:
            setattr(core_module, "colored", _tc_colored)
        else:
            setattr(core_module, "colored", lambda s, *_a, **_k: _ansi_stripper(s))
    except Exception:
        if ansi_ok:
            setattr(core_module, "colored", lambda s, *_a, **_k: str(s))
        else:
            setattr(core_module, "colored", lambda s, *_a, **_k: _ansi_stripper(s))

_ensure_colored(core)

if hasattr(cfg, "validate_config") and callable(cfg.validate_config):
    try:
        cfg.validate_config()
    except Exception as e:
        print(f"[WARN] config.validate_config() raised: {e}")

def _inject_footer_context(core_module, org_name, version_string):
    ts = datetime.now().astimezone().strftime("%Y-%m-%d %H:%M:%S %Z")
    setattr(core_module, "org_name", org_name)
    setattr(core_module, "version_string", version_string)
    setattr(core_module, "report_timestamp", ts)

def _resolve_into_config(cfg_module):
    mode = getattr(cfg_module, "mode", None) or getattr(cfg_module, "MODE", None) \
           or os.environ.get("PYTHEIA_MODE") or "azel"
    setattr(cfg_module, "mode", mode)

    input_dat = (
        getattr(cfg_module, "input_dat", None)
        or getattr(cfg_module, "DATA_FILE", None)
        or os.environ.get("PYTHEIA_DATA")
    )
    if len(sys.argv) > 1 and sys.argv[1] and sys.argv[1] != "--":
        input_dat = sys.argv[1]
    if not input_dat:
        raise ValueError(
            "No data file specified.\n"
            "Set input_dat (or DATA_FILE) in config.py, or pass a path as argument:\n"
            "    python Pytheia100_main.py path/to/data.dat\n"
            "Or set environment variable PYTHEIA_DATA."
        )
    setattr(cfg_module, "input_dat", os.path.abspath(input_dat))

    output_dir = getattr(cfg_module, "output_dir", None) or getattr(cfg_module, "OUTPUT_DIR", None) \
                 or os.environ.get("PYTHEIA_OUTPUT_DIR") or "My_Pointing_model_results"
    setattr(cfg_module, "output_dir", output_dir)
    Path(output_dir).mkdir(parents=True, exist_ok=True)

    if not getattr(cfg_module, "selected_terms", None):
        setattr(cfg_module, "selected_terms", ["AN", "AW", "IA", "IE", "CA", "TX", "ECES", "HZSZ2"])

def _apply_config_to_core(core_module, cfg_module, isolate_outputs=True) -> str:
    selected_terms = list(getattr(cfg_module, "selected_terms", []) or [])
    input_dat      = getattr(cfg_module, "input_dat", None)
    output_dir     = getattr(cfg_module, "output_dir", ".")
    mode           = getattr(cfg_module, "mode", None)
    lat            = getattr(cfg_module, "lat", None)
    phi_val        = _derive_phi_from_lat(lat)
    aux1           = getattr(cfg_module, "aux1", None)
    aux2           = getattr(cfg_module, "aux2", None)

    if isolate_outputs:
        output_dir = _unique_output_dir(output_dir, selected_terms)

    overwrite_map = {
        "selected_terms": selected_terms,
        "input_dat": input_dat,
        "output_dir": output_dir,
        "mode": mode,
        "lat": lat,
        "phi": phi_val,
        "aux1": aux1,
        "aux2": aux2,
    }
    for name, value in overwrite_map.items():
        try:
            setattr(core_module, name, value)
        except Exception:
            pass

    try:
        cfg_module.output_dir = output_dir
    except Exception:
        pass

    if hasattr(cfg_module, "pointing_model") and callable(cfg_module.pointing_model):
        def _pm(c1, c2, terms, aux1=None, aux2=None):
            return cfg_module.pointing_model(c1, c2, terms, aux1=aux1, aux2=aux2)
        setattr(core_module, "pointing_model", _pm)

    return output_dir

def _call_core(core_module, cfg_module):
    if hasattr(core_module, "main") and callable(core_module.main):
        sig = inspect.signature(core_module.main)
        params = list(sig.parameters.keys())
        argmap = {}
        phi_val = _derive_phi_from_lat(getattr(cfg_module, "lat", None))
        for name in params:
            if name in ("input_file","input_dat","input_path"):
                argmap[name] = getattr(cfg_module, "input_dat", None)
            elif name in ("output_dir","out_dir"):
                argmap[name] = getattr(core_module, "output_dir", getattr(cfg_module, "output_dir", None))
            elif name in ("selected_terms","terms"):
                argmap[name] = getattr(core_module, "selected_terms", getattr(cfg_module, "selected_terms", None))
            elif name == "mode":
                argmap[name] = getattr(core_module, "mode", getattr(cfg_module, "mode", None))
            elif name == "lat":
                argmap[name] = getattr(core_module, "lat", getattr(cfg_module, "lat", None))
            elif name == "phi":
                argmap[name] = getattr(core_module, "phi", phi_val)
            elif name == "aux1":
                argmap[name] = getattr(core_module, "aux1", getattr(cfg_module, "aux1", None))
            elif name == "aux2":
                argmap[name] = getattr(core_module, "aux2", getattr(cfg_module, "aux2", None))
            elif name in ("config","cfg"):
                argmap[name] = cfg_module
        return core_module.main(**argmap)
    return None

effective_outdir = getattr(cfg, "output_dir", ".")
with seat_lock():
    try:
        _resolve_into_config(cfg)
        core = importlib.import_module("Pytheia100_core")
        effective_outdir = _apply_config_to_core(core, cfg, isolate_outputs=True)

        _terms = list(getattr(core, "selected_terms", []) or [])
        print(f"[INFO] {PRODUCT} v{VERSION} | Org: {_org}")
        print(f"[INFO] Mode: {getattr(core,'mode','azel')} | Data: {os.path.basename(getattr(core,'input_dat','<unset>'))}")
        print(f"[INFO] Selected terms ({len(_terms)}): {_terms}")
        print(f"[INFO] Output directory: {effective_outdir}")

        audit_event("startup", output_dir=effective_outdir)
        _inject_footer_context(core, org_name=_org, version_string=f"{PRODUCT} v{VERSION}")

        if hasattr(cfg, "pointing_model") and callable(cfg.pointing_model):
            print("[INFO] Using custom pointing_model from config.py")

        _ = _call_core(core, cfg)
        sys.stdout.flush()

        try:
            mode_disp = (getattr(core, "mode", "azel") or "azel").upper()
            msg = (
                f"\nPointing model ({mode_disp} mode) successfully computed and saved. "
                f"Check output directory for results!"
            )
            try:
                from colorama import Fore, Style
                print(Fore.GREEN + msg + Style.RESET_ALL)
            except Exception:
                print(core.colored(msg, "green") if hasattr(core, "colored") else msg)
        except Exception:
            pass

        audit_event("run-complete", output_dir=effective_outdir)
        html_path = os.path.join(effective_outdir, "combined_results_bayesian.html")

        print(f"[SUCCESS] {PRODUCT} v{VERSION}: computation finished.")
        print(f"[INFO] Report: {html_path}")

    except Exception as e:
        try:
            audit_event(f"run-error: {type(e).__name__}: {e}", output_dir=effective_outdir)
        finally:
            print("\n[ERROR] Failed to compute pointing model:")
            print(f" * Error: {e}")
            tb = traceback.extract_tb(sys.exc_info()[2])[-1] if sys.exc_info()[2] else None
            if tb:
                print(f" * File: {tb.filename}")
                print(f" * Line: {tb.lineno}")
                print(f" * Code: {tb.line}")
            sys.exit(3)
