# pytheia_guard.py — production guard (Ed25519 verify + integrity + seat locking + near-expiry warnings)
import os, json, time, base64, hashlib, tempfile, getpass, socket, platform
from pathlib import Path
from contextlib import contextmanager

# Ed25519 (public-key verify) — requires cryptography
from cryptography.hazmat.primitives.serialization import load_pem_public_key
from cryptography.hazmat.primitives.asymmetric.ed25519 import Ed25519PublicKey
from cryptography.exceptions import InvalidSignature

_G_STATE = {"license": None}
_LOCK_DIR = Path(tempfile.gettempdir()) / "pytheia_locks"

def _now_utc():
    return time.strftime("%Y-%m-%dT%H:%M:%SZ", time.gmtime())

def compute_machine_hash():
    """Simple machine fingerprint; issuer can bind license to this string."""
    import uuid
    user = getpass.getuser()
    host = socket.gethostname()
    plat = platform.platform()
    mac = 0
    try:
        mac = uuid.getnode()
    except Exception:
        pass
    raw = f"{user}|{host}|{plat}|{mac}".encode("utf-8")
    return hashlib.sha256(raw).hexdigest()

def _canonical_payload_bytes(payload: dict) -> bytes:
    # Stable JSON encoding for signing/verification (licenses)
    return json.dumps(payload, sort_keys=True, separators=(",", ":")).encode("utf-8")

def verify_code_integrity(path: str, expected_sha256: str):
    """Compare file SHA-256 to expected. Raise on mismatch."""
    h = hashlib.sha256()
    with open(path, "rb") as f:
        for chunk in iter(lambda: f.read(8192), b""):
            h.update(chunk)
    actual = h.hexdigest()
    if expected_sha256 and actual != expected_sha256:
        raise RuntimeError(f"Code integrity check failed for {path}. Expected {expected_sha256} got {actual}")
    return actual

def verify_manifest_signature(manifest_path: str, sig_path: str, public_key_path: str) -> dict:
    """
    Verify Ed25519 signature over the *raw bytes* of INTEGRITY.json.
    Returns parsed JSON on success; raises with a clear error if invalid.
    """
    data = Path(manifest_path).read_bytes()  # sign/verify exact bytes (no reformatting)
    sig_b64 = Path(sig_path).read_text(encoding="utf-8").strip()
    sig = base64.b64decode(sig_b64)

    pub = load_pem_public_key(Path(public_key_path).read_bytes())
    if not isinstance(pub, Ed25519PublicKey):
        raise RuntimeError("Public key is not an Ed25519 key.")
    try:
        pub.verify(sig, data)
    except InvalidSignature as e:
        raise RuntimeError(
            "INTEGRITY.sig is invalid. The integrity manifest was not signed by the matching private key, "
            "or the manifest was altered."
        ) from e

    # Parse JSON (accepts files written without BOM; if you ever use BOM, write with utf-8-sig consistently)
    try:
        return json.loads(data.decode("utf-8"))
    except Exception as e:
        raise RuntimeError("INTEGRITY.json is not valid UTF-8 JSON.") from e

def audit_event(event: str, output_dir: str = "."):
    Path(output_dir).mkdir(parents=True, exist_ok=True)
    path = Path(output_dir) / "pytheia_audit.jsonl"
    rec = {"ts": _now_utc(), "event": event}
    lic = _G_STATE.get("license") or {}
    if lic:
        rec.update({"org": lic.get("org"), "license_id": lic.get("license_id"), "product": lic.get("product")})
    with open(path, "a", encoding="utf-8") as f:
        f.write(json.dumps(rec) + "\n")
    return str(path)

def watermark_text(extra: str = "") -> str:
    lic = _G_STATE.get("license") or {}
    parts = ["PYTHEIA", _now_utc()]
    if lic:
        parts += [lic.get("org",""), lic.get("product",""), lic.get("license_id",""), lic.get("expires_at","")]
    if extra:
        parts.append(extra)
    return " | ".join([p for p in parts if p])

def license_boot(*, license_path: str, public_key_path: str, expected_product: str,
                 expected_org: str | None = None, check_machine_hash: bool = False,
                 warn_before_days: float | None = None):
    """Load and verify license (Ed25519). Stores state for later checks."""
    from datetime import datetime, timezone

    data = json.loads(Path(license_path).read_text(encoding="utf-8"))
    payload = data["payload"]
    sig_b64 = data["signature"]

    # Verify license signature
    pub = load_pem_public_key(Path(public_key_path).read_bytes())
    if not isinstance(pub, Ed25519PublicKey):
        try:
            pub = Ed25519PublicKey.from_public_bytes(pub.public_bytes())
        except Exception:
            raise RuntimeError("Public key is not an Ed25519 key.")
    sig = base64.b64decode(sig_b64)
    try:
        pub.verify(sig, _canonical_payload_bytes(payload))
    except InvalidSignature as e:
        raise RuntimeError(
            "License signature is invalid. "
            "Ensure the license was issued with the private key matching the shipped pytheia_public_key.pem, "
            "and the JSON file was not altered."
        ) from e

    # Claims
    if payload.get("product") != expected_product:
        raise RuntimeError("License product mismatch.")
    if expected_org is not None and payload.get("org") != expected_org:
        raise RuntimeError("License organization mismatch.")

    # Expiry
    expires_at = payload.get("expires_at")  # ISO8601 Z
    now_iso = _now_utc()
    if expires_at and now_iso > expires_at:
        raise RuntimeError("License expired.")

    # Optional machine binding
    if check_machine_hash:
        mh = payload.get("machine_hash")
        local = compute_machine_hash()
        if mh and mh != local:
            raise RuntimeError("License is bound to another machine.")

    # Save state
    _G_STATE["license"] = {
        "org": payload.get("org"),
        "product": payload.get("product"),
        "version": payload.get("version"),
        "license_id": payload.get("license_id"),
        "expires_at": payload.get("expires_at"),
        "seats": int(payload.get("seats", 1)),
    }

    # Near-expiry soft warning
    if warn_before_days is None:
        env_val = os.getenv("PYTHEIA_WARN_DAYS", "").strip()
        try:
            warn_before_days = float(env_val) if env_val else 7.0
        except Exception:
            warn_before_days = 7.0
    try:
        if expires_at:
            from datetime import datetime, timezone
            dt_exp = datetime.strptime(expires_at, "%Y-%m-%dT%H:%M:%SZ").replace(tzinfo=timezone.utc)
            dt_now = datetime.strptime(now_iso, "%Y-%m-%dT%H:%M:%SZ").replace(tzinfo=timezone.utc)
            days_left = (dt_exp - dt_now).total_seconds() / 86400.0
            _G_STATE["license"]["_days_left"] = days_left
            if 0 < days_left < float(warn_before_days):
                print(f"[WARNING] License for {payload.get('org','(unknown org)')} "
                      f"expires in {days_left:.1f} days ({expires_at}).\n"
                      f"          Contact support@orionisgroup.com for renewal.")
                try:
                    audit_event(f"license-near-expiry: {days_left:.1f} days remaining")
                except Exception:
                    pass
    except Exception:
        pass

    return _G_STATE["license"]

@contextmanager
def seat_lock():
    """Single/multi-seat lock derived from license_id; allows parallel <= seats."""
    lic = _G_STATE.get("license") or {}
    if not lic:
        raise RuntimeError("No license loaded.")
    lid = lic.get("license_id") or "PYTHEIA-UNK"
    seats = max(1, int(lic.get("seats", 1)))
    Path(_LOCK_DIR).mkdir(parents=True, exist_ok=True)
    lock_path = None
    for slot in range(seats):
        candidate = Path(_LOCK_DIR) / f"{lid}.{slot}.lock"
        try:
            fd = os.open(candidate, os.O_CREAT | os.O_EXCL | os.O_WRONLY)
            os.write(fd, str(os.getpid()).encode("utf-8"))
            os.close(fd)
            lock_path = candidate
            break
        except FileExistsError:
            continue
    if lock_path is None:
        raise RuntimeError("All licensed seats are currently in use.")
    try:
        yield
    finally:
        try:
            os.remove(lock_path)
        except FileNotFoundError:
            pass
